//////////////////////////////////////////////////////////
//
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
//  Structures to pass data to and from the OCP Channel
//  for both the TL1 and TL2 OCP specific channels.
//
// Joe Chou, Sonics Inc.
// Alan Kamas, for Sonics Inc.
// Yann Bajot, Prosilog
// Stephane Guntz, Prosilog
//
// 10/01/2003 Original revision
// 05/14/2004 Added ==, =, << operators for Request,
//            DataHS, and Response structures
//////////////////////////////////////////////////////////

#ifndef _OCP_GLOBALS_H
#define _OCP_GLOBALS_H

#include <string>
#include <assert.h>

#include <systemc.h>

// needed for gcc 3.3:
using namespace std;

// -------------------------------------------------
// OCP 2.0
// -------------------------------------------------

// MCmd encoding
enum OCPMCmdType {
    OCP_MCMD_IDLE = 0,
    OCP_MCMD_WR,
    OCP_MCMD_RD,
    OCP_MCMD_RDEX,
    OCP_MCMD_RDL,
    OCP_MCMD_WRNP,
    OCP_MCMD_WRC,
    OCP_MCMD_BCST
};

// SResp encoding
enum OCPSRespType {
    OCP_SRESP_NULL = 0,
    OCP_SRESP_DVA,
    OCP_SRESP_FAIL,
    OCP_SRESP_ERR
};

// MBurstSeq encoding
enum OCPMBurstSeqType {
    OCP_MBURSTSEQ_INCR = 0,
    OCP_MBURSTSEQ_DFLT1,
    OCP_MBURSTSEQ_WRAP,
    OCP_MBURSTSEQ_DFLT2,
    OCP_MBURSTSEQ_XOR,
    OCP_MBURSTSEQ_STRM,
    OCP_MBURSTSEQ_UNKN,
    OCP_MBURSTSEQ_RESERVED
};


/////////////////////////////////////////////////////////////
// DataHSGrp holds the dataflow signals for a data hand shake
// TL1 Only
/////////////////////////////////////////////////////////////
template <class Td>
class OCPDataHSGrp
{
  public:
    typedef Td DataType;

  public:
    //
    DataType     MData;
    unsigned int MDataThreadID;
    unsigned int MDataByteEn;
    unsigned int MDataInfo;

    // OCP 2.0 Burst
    bool         MDataLast;

    // synchronization signal
    bool         MDataValid;

    // constructor
    OCPDataHSGrp(void) { reset(); }

    // define a copy constructor
    OCPDataHSGrp(const OCPDataHSGrp& src)
      : MData(src.MData),
        MDataThreadID(src.MDataThreadID),
        MDataByteEn(src.MDataByteEn),
        MDataInfo(src.MDataInfo),
        MDataLast(src.MDataLast),
        MDataValid(src.MDataValid)
    {}

    // define a copy assignment
    OCPDataHSGrp& operator=(const OCPDataHSGrp& rhs)
    {
        if (this == &rhs) { return *this; }
        copy(rhs);
        return *this;
    }

    // define a explicit copy
    void copy(const OCPDataHSGrp& src)
    {
        MData = src.MData;
        MDataThreadID = src.MDataThreadID;
        MDataByteEn = src.MDataByteEn;
        MDataInfo = src.MDataInfo;
        MDataLast = src.MDataLast;
        MDataValid = src.MDataValid;
    }

    void reset(void)
    {
        MData = 0;
        MDataThreadID = 0;
        MDataByteEn = 0xFFFFFFFF;
        MDataInfo = 0;
        MDataLast = true;
        MDataValid = false;
    }

    string print(void) { assert(0); }

	//operator overloading, so that class can be passed through ports 
	inline bool operator == (const OCPDataHSGrp& rhs) const {
		return (rhs.MData == MData
				&& MDataThreadID == rhs.MDataThreadID  
				&& MDataByteEn == rhs.MDataByteEn  
				&& MDataInfo == rhs.MDataInfo  
				&& MDataLast == rhs.MDataLast  
				&& MDataValid == rhs.MDataValid);
	}

	inline friend ostream& operator << ( ostream& os, OCPDataHSGrp const & v)  {
//		os << " -MData: "<<v.MData << " -MDataThreadID: "<<v.MDataThreadID << " -MDataByteEn: "<<v.MDataByteEn << " -MDataInfo: "<<v.MDataInfo<< " -MDataLast: " v.MDataLast<< " -MDataValid: "<<v.MDataValid<< " ";
//		os << "toto" ;
		return os;
	}

	inline friend void sc_trace( sc_trace_file *tf, const OCPDataHSGrp& v, const sc_string& NAME)   {
		sc_trace(tf, v.MData, NAME+".MData");
		sc_trace(tf, v.MDataThreadID, NAME+".MDataThreadID");
		sc_trace(tf, v.MDataByteEn, NAME+".MDataByteEn");
		sc_trace(tf, v.MDataInfo, NAME+".MDataInfo");
		sc_trace(tf, v.MDataLast, NAME+".MDataLast");
		sc_trace(tf, v.MDataValid, NAME+".MDataValid");	
	}
};

///////////////////////////////////////////////
// RequestGrp holds the OCP request information 
///////////////////////////////////////////////
template <class Td, class Ta>
class OCPRequestGrp
{
  public:
    typedef Td DataType;
    typedef Ta AddrType;

  public:
    //
    OCPMCmdType   MCmd;
    AddrType      MAddr;
    unsigned int  MAddrSpace;
    unsigned int  MByteEn;
    unsigned int  MThreadID;
    unsigned int  MConnID;
    unsigned int  MReqInfo;

    // for OCP 2.0 Burst
    unsigned int     MAtomicLength;
    unsigned int     MBurstLength;
    bool             MBurstPrecise;
    OCPMBurstSeqType MBurstSeq;
    bool             MBurstSingleReq;
    bool             MReqLast;

    // request group with data?
    bool          HasMData;
    DataType      MData;        // TL1 Only     
    unsigned int  MDataInfo;    
    DataType      *MDataPtr;    // TL2 Only

    // constructor
    OCPRequestGrp(bool has_mdata = true) { reset(has_mdata); }

    // define a copy constructor
    OCPRequestGrp(const OCPRequestGrp& src)
      : MCmd(src.MCmd),
        MAddr(src.MAddr),
        MAddrSpace(src.MAddrSpace),
        MByteEn(src.MByteEn),
        MThreadID(src.MThreadID),
        MConnID(src.MConnID),
        MReqInfo(src.MReqInfo),
        // MBurst(src.MBurst),  // Old OCP1.0 - no longer supported
        MAtomicLength(src.MAtomicLength),
        MBurstLength(src.MBurstLength),
        MBurstPrecise(src.MBurstPrecise),
        MBurstSeq(src.MBurstSeq),
        MBurstSingleReq(src.MBurstSingleReq),
        MReqLast(src.MReqLast),
        HasMData(src.HasMData),
        MData(src.MData),
        MDataInfo(src.MDataInfo),
        MDataPtr(src.MDataPtr)
    {}

    // define a copy assignment
    OCPRequestGrp& operator= (const OCPRequestGrp& rhs)
    {
        if (this == &rhs) { return *this; }
        copy(rhs);
        return *this;
    }

    // define a explicit copy
    void copy(const OCPRequestGrp& src)
    {
        MCmd = src.MCmd;
        MAddr = src.MAddr;
        MAddrSpace = src.MAddrSpace;
        MByteEn = src.MByteEn;
        MThreadID = src.MThreadID;
        MConnID = src.MConnID;
        MReqInfo = src.MReqInfo;
        // MBurst = src.MBurst; // Old OCP 1.0 - no longer supported
        MAtomicLength = src.MAtomicLength;
        MBurstLength = src.MBurstLength;
        MBurstPrecise = src.MBurstPrecise;
        MBurstSeq = src.MBurstSeq;
        MBurstSingleReq = src.MBurstSingleReq;
        MReqLast = src.MReqLast;
        HasMData = src.HasMData;
        MData = src.MData;
        MDataInfo = src.MDataInfo;
        MDataPtr = src.MDataPtr;
    };

    void reset(bool has_mdata = true)
    {
        MCmd = OCP_MCMD_IDLE;
        MAddr = 0;
        MAddrSpace = 0;
        MByteEn = 0xFFFFFFFF;
        MThreadID = 0;
        MConnID = 0;
        MReqInfo = 0;
        // MBurst = OCP_MBURST_LAST; // Old OCP 1.0 - no longer supported
        MAtomicLength = 1;
        MBurstLength = 1;
        MBurstPrecise = true;
        MBurstSeq = OCP_MBURSTSEQ_INCR;
        MBurstSingleReq = false;
        MReqLast = true;
        HasMData = has_mdata;
        MData = 0;
        MDataInfo = 0;
        MDataPtr = NULL;
    }

    string print(void) const { assert(0); }

	//operator overloading, so that class can be passed through ports 
	inline bool operator == (const OCPRequestGrp& rhs) const {
		return ( MCmd == rhs.MCmd
				&& MAddr == rhs.MAddr
				&& MAddrSpace == rhs.MAddrSpace
				&& MByteEn == rhs.MByteEn
				&& MThreadID == rhs.MThreadID
				&& MConnID == rhs.MConnID
				&& MReqInfo == rhs.MReqInfo
				&& MAtomicLength == rhs.MAtomicLength
				&& MBurstLength == rhs.MBurstLength
				&& MBurstPrecise == rhs.MBurstPrecise
				&& MBurstSeq == rhs.MBurstSeq
				&& MBurstSingleReq == rhs.MBurstSingleReq
				&& MReqLast == rhs.MReqLast
				&& HasMData == rhs.HasMData
				&& MData == rhs.MData
				&& MDataInfo == rhs.MDataInfo
				&& MDataPtr == rhs.MDataPtr);
	}

	inline friend ostream& operator << ( ostream & os, OCPRequestGrp const & v)   {
		os << " -MCmd: "<<v.MCmd << " -MAddr: "<<v.MAddr << " -MAddrSpace: "<<v.MAddrSpace
			<< " -MByteEn: "<<v.MByteEn <<" -MThreadID: "<<v.MThreadID<<	" -MConnID: "<<v.MConnID
			<<" -MReqInfo: "<<v.MReqInfo<<" -MAtomicLength: "<<v.MAtomicLength
			<<" -MBurstLength: "<<v.MBurstLength<<" -MBurstPrecise: "<<v.MBurstPrecise
			<<" -MBurstSeq: "<<v.MBurstSeq<<" -MBurstSingleReq: "<<v.MBurstSingleReq
			<<" -MReqLast: "<<v.MReqLast<<" -HasMData: "<<v.HasMData
			<<" -MData: "<<v.MData<<" -MDataInfo: "<<v.MDataInfo
			<<" -MDataPtr: "<<v.MDataPtr<< " ";
		return os;
	}

	inline friend void sc_trace( sc_trace_file *tf, const OCPRequestGrp& v, const sc_string& NAME)   {
		sc_trace(tf, v.MCmd, NAME+".MCmd");
		sc_trace(tf, v.MAddr, NAME+".MAddr");
		sc_trace(tf, v.MAddrSpace, NAME+".MAddrSpace");
		sc_trace(tf, v.MByteEn, NAME+".MByteEn");
		sc_trace(tf, v.MConnID, NAME+".MConnID");
		sc_trace(tf, v.MReqInfo, NAME+".MReqInfo");	
		sc_trace(tf, v.MAtomicLength, NAME+".MAtomicLength");
		sc_trace(tf, v.MBurstLength, NAME+".MBurstLength");
		sc_trace(tf, v.MBurstPrecise, NAME+".MBurstPrecise");
		sc_trace(tf, v.MBurstSeq, NAME+".MBurstSeq");
		sc_trace(tf, v.MBurstSingleReq, NAME+".MBurstSingleReq");
		sc_trace(tf, v.MReqLast, NAME+".MReqLast");	
		sc_trace(tf, v.HasMData, NAME+".HasMData");
		sc_trace(tf, v.MData, NAME+".MData");
		sc_trace(tf, v.MDataInfo, NAME+".MDataInfo");
		sc_trace(tf, v.MDataPtr, NAME+".MDataPtr");	
	}
};

/////////////////////////////////////////////////
// ResponseGrp holds the OCP response information
/////////////////////////////////////////////////
template <class Td>
class OCPResponseGrp
{
  public:
    typedef Td DataType;

  public:
    OCPSRespType SResp;
    DataType     SData;         // TL1 Only
    DataType     *SDataPtr;     // TL2 Only
    unsigned int SDataInfo;
    unsigned int SThreadID;
    unsigned int SRespInfo;
    bool         SRespLast;

    // constructor
    OCPResponseGrp(void) { reset(); }
 
    // define a copy constructor
    OCPResponseGrp(const OCPResponseGrp& src)
      : SResp(src.SResp),
        SData(src.SData),
        SDataPtr(src.SDataPtr),
        SDataInfo(src.SDataInfo),
        SThreadID(src.SThreadID),
        SRespInfo(src.SRespInfo),
        SRespLast(src.SRespLast)
    {}
 
    // define a copy assignment
    OCPResponseGrp& operator= (const OCPResponseGrp& rhs)
    {
        if (this == &rhs) { return *this; }
        copy(rhs);
        return *this;
    }

    // define a explicit copy
    void copy(const OCPResponseGrp& src)
    {
        SResp = src.SResp;
        SData = src.SData;
        SDataPtr = src.SDataPtr;
        SDataInfo = src.SDataInfo;
        SThreadID = src.SThreadID;
        SRespInfo = src.SRespInfo;
        SRespLast = src.SRespLast;
    }

    void reset(void)
    {
        SResp = OCP_SRESP_NULL;
        SData = 0;
        SDataPtr = NULL;
        SDataInfo = 0;
        SThreadID = 0;
        SRespInfo = 0;
        SRespLast = true;
    }
    string print(void) const { assert(0); }

	//operator overloading, so that class can be passed through ports 
	inline bool operator == (const OCPResponseGrp& rhs) const {
		return ( SResp == rhs.SResp
				&& SData == rhs.SData
				&& SDataPtr == rhs.SDataPtr
				&& SDataInfo == rhs.SDataInfo
				&& SThreadID == rhs.SThreadID
				&& SRespInfo == rhs.SRespInfo
				&& SRespLast == rhs.SRespLast);
	}

	inline friend ostream& operator << ( ostream& os, OCPResponseGrp const & v)  {
		os << " -SResp: "<<v.SResp << " -SData: "<<v.SData << " -SDataPtr: "<<v.SDataPtr
			<< " -SDataInfo: "<<v.SDataInfo <<" -SThreadID: "<<v.SThreadID
			<<" -SRespInfo: "<<v.SRespInfo	<<" -SRespLast: "<<v.SRespLast<< " ";
		return os;
	}

	inline friend void sc_trace( sc_trace_file *tf, const OCPResponseGrp& v, const sc_string& NAME)   {
		sc_trace(tf, v.SResp, NAME+".SResp");
		sc_trace(tf, v.SData, NAME+".SData");
		sc_trace(tf, v.SDataPtr, NAME+".SDataPtr");
		sc_trace(tf, v.SDataInfo, NAME+".SDataInfo");
		sc_trace(tf, v.SThreadID, NAME+".SThreadID");
		sc_trace(tf, v.SRespInfo, NAME+".SRespInfo");	
		sc_trace(tf, v.SRespLast, NAME+".SRespLast");
	}
};

//=====================================================//
// Code Below this line used by the TL2 Channel only


/////////////////////////////////////////////////
// SideBandGrp holds the OCP sideband information
// TL2 Channel Only
/////////////////////////////////////////////////

class OCPSidebandGrp {
  public:

    // reset signals
    bool MReset_n;
    bool SReset_n;

    // system/core signals
    unsigned int  Control;
    bool ControlWr;
    bool ControlBusy;
    unsigned int  Status;
    bool StatusRd;
    bool StatusBusy;

    // sideband signals
    unsigned int  MFlag;
    bool MError;
    unsigned int  SFlag;
    bool SError;
    bool SInterrupt;


    void reset()
    {
      SInterrupt = false;
      SError = false;
      MError = false;
      MFlag = 0;
      SFlag = 0;
      Control = 0;;
      ControlWr = false;
      ControlBusy = false;
      Status = 0;
      StatusRd = false;
      StatusBusy = false;
    }

    OCPSidebandGrp()
    {
      MReset_n = true;
      SReset_n = true;
      reset();
    }

	//operator overloading, so that class can be passed through ports 
	inline bool operator == (const OCPSidebandGrp& rhs) const {
		return ( MReset_n == rhs.MReset_n
				&& SReset_n == rhs.SReset_n
				&& Control == rhs.Control
				&& ControlWr == rhs.ControlWr
				&& ControlBusy == rhs.ControlBusy
				&& Status == rhs.Status
				&& StatusRd == rhs.StatusRd
				&& StatusBusy == rhs.StatusBusy
				&& MFlag == rhs.MFlag
				&& SFlag == rhs.SFlag
				&& MError == rhs.MError
				&& SError == rhs.SError
				&& SInterrupt == rhs.SInterrupt);
	}

	//define a copy operator
	inline OCPSidebandGrp& operator = (const OCPSidebandGrp& rhs)   {
        if (this == &rhs) { return *this; }
		MReset_n = rhs.MReset_n;
		SReset_n = rhs.SReset_n;
		Control = rhs.Control;
		ControlWr = rhs.ControlWr;
		ControlBusy = rhs.ControlBusy;
		Status = rhs.Status;
		StatusRd = rhs.StatusRd;
		StatusBusy = rhs.StatusBusy;
		MFlag = rhs.MFlag;
		SFlag = rhs.SFlag;
		MError = rhs.MError;
		SError = rhs.SError;
		SInterrupt = rhs.SInterrupt;
        return *this;
	}

	inline friend ostream& operator << ( ostream& os, OCPSidebandGrp const & v)  {
		os << " -MReset_n: "<<v.MReset_n << " -SReset_n: "<<v.SReset_n
			<< " -Control: "<<v.Control<< " -ControlWr: "<<v.ControlWr 
			<<" -ControlBusy: "<<v.ControlBusy<<" -Status: "<<v.Status
			<<" -StatusRd: "<<v.StatusRd<<" -StatusBusy: "<<v.StatusBusy
			<<" -MFlag: "<<v.MFlag<<" -SFlag: "<<v.SFlag
			<<" -MError: "<<v.MError<<" -SError: "<<v.SError
			<<" -SInterrupt: "<<v.SInterrupt<< " ";
		return os;
	}

	inline friend void sc_trace( sc_trace_file *tf, const OCPSidebandGrp& v, const sc_string& NAME)   {
		sc_trace(tf, v.MReset_n, NAME+".MReset_n");
		sc_trace(tf, v.SReset_n, NAME+".SReset_n");
		sc_trace(tf, v.Control, NAME+".Control");
		sc_trace(tf, v.ControlWr, NAME+".ControlWr");
		sc_trace(tf, v.ControlBusy, NAME+".ControlBusy");
		sc_trace(tf, v.Status, NAME+".Status");
		sc_trace(tf, v.StatusRd, NAME+".StatusRd");
		sc_trace(tf, v.StatusBusy, NAME+".StatusBusy");
		sc_trace(tf, v.MFlag, NAME+".MFlag");
		sc_trace(tf, v.SFlag, NAME+".SFlag");
		sc_trace(tf, v.MError, NAME+".MError");
		sc_trace(tf, v.SError, NAME+".SError");
		sc_trace(tf, v.SInterrupt, NAME+".SInterrupt");	
	}
};

/////////////////////////////////////////////////
// ThreadBusyGrp holds the OCP Threadbusy information
// TL2 Channel Only
/////////////////////////////////////////////////

class OCPThreadBusyGrp {
  public:

    // Threadbusy signals
    unsigned int MThreadBusy;
    unsigned int MDataThreadBusy;
    unsigned int SThreadBusy;

    void reset()
    {
      MThreadBusy = false;
      MDataThreadBusy = false;
      SThreadBusy = false;
    }

    OCPThreadBusyGrp()
    {
      reset();
    }

	//operator overloading, so that class can be passed through ports 
	inline bool operator == (const OCPThreadBusyGrp& rhs) const {
		return ( MThreadBusy == rhs.MThreadBusy
				&& MDataThreadBusy == rhs.MDataThreadBusy
				&& SThreadBusy == rhs.SThreadBusy);
	}

	//define a copy operator
	inline OCPThreadBusyGrp& operator = (const OCPThreadBusyGrp& rhs)   {
        if (this == &rhs) { return *this; }
		MThreadBusy = rhs.MThreadBusy;
		MDataThreadBusy = rhs.MDataThreadBusy;
		SThreadBusy = rhs.SThreadBusy;
        return *this;
	}

	inline friend ostream& operator << ( ostream& os, OCPThreadBusyGrp const & v)  {
		os << " -MThreadBusy: "<<v.MThreadBusy << " -MDataThreadBusy: "<<v.MDataThreadBusy
			<< " -SThreadBusy: "<<v.SThreadBusy<< " ";
		return os;
	}

	inline friend void sc_trace( sc_trace_file *tf, const OCPThreadBusyGrp& v, const sc_string& NAME)   {
		sc_trace(tf, v.MThreadBusy, NAME+".MThreadBusy");
		sc_trace(tf, v.MDataThreadBusy, NAME+".MDataThreadBusy");
		sc_trace(tf, v.SThreadBusy, NAME+".SThreadBusy");	
	}
};

#endif // _OCP_GLOBALS_H
