// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 03/10/2003  
//
//  Description : Layer Adapter, TL0-TL1 example, Synchronous TL0 OCP Slave 
//    Features:
//    - TL0 slave implementing OCP1.0 Basic Signals 
//    - Synchronous Response, with propagation delay emulation
//    - Synchronous Request Acknowldege, with propagation delay emulation  
//    
//  Parameter    :
//    Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - sc_time RequestDelay:
//        delay between MCmd assertion and SCmdAccept assertion
//     - sc_time ResponseDelay:
//        delay between Clock rising edge and Response Group signals assertion
// ============================================================================

#include "ocp_tl0_slave_sync2.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL0_Slave_sync2::OCP_TL0_Slave_sync2
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Slave_sync2<TdataCl>::OCP_TL0_Slave_sync2
       (
          sc_module_name name_
          , sc_time RequestDelay
          , sc_time ResponseDelay
        )
          : sc_module(name_)
          , m_RequestDelay(RequestDelay) 
          , m_ResponseDelay(ResponseDelay)
          , m_request_fifo(REQUEST_FIFO_DEPTH)
{
  
  // Output ports initializing
  SResp.initialize(sc_bv<2>(OCP_SRESP_NULL));
  SData.initialize(Td(0));
  SCmdAccept.initialize('0');
  
  // Request Processes
  SC_THREAD(MCmdThread);
  sensitive << MCmd;
  
  SC_METHOD(SCmdAcceptSetThread);
  dont_initialize();
  sensitive << e_SCmdAccept_Set ;
  
  SC_METHOD(SCmdAcceptResetThread);
  sensitive_pos << Clk;

  // Response Processes
  SC_THREAD(ExecAndSendResponse);
  sensitive_pos << Clk;
   
} // end of constructor


// ----------------------------------------------------------------------------
// Process : OCP_TL0_Slave_sync2::~OCP_TL0_Slave_sync2
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Slave_sync2<TdataCl>::~OCP_TL0_Slave_sync2()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_sync2::MCmdThread() 
//
//  Catch 'MCmd' variations , push OCP request signals on the request fifo
//  , wait the end of cycle and notify 'SCmdAccept' Thread (with a delay "m_RequestDelay")
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_sync2<TdataCl>::MCmdThread()
{
  while(1) {
    wait();  // wait for 'Mcmd' value change

    if(MCmd.read() != sc_bv<3>( (OCPMCmdType)OCP_MCMD_IDLE))
    {
      // get Request signals 
      request_struct<Td,Ta> req;
      req.mcmd = MCmd.read().to_uint();
      req.mdata = MData.read();
      req.maddr = MAddr.read();

#ifdef DEBUG_G1
      cout << "TL0 Slave got request "
        << " delta " << simcontext()->delta_count()
        << " time " << sc_time_stamp().to_seconds()
        << " data " << req.maddr << endl;
#endif

      // push request 
      m_request_fifo.write(req);

      // wait one cycle 
      wait(Clk.posedge_event());

      // notify event 'e_SCmdAccept_Set' 
      e_SCmdAccept_Set.notify(m_RequestDelay);

      // wait end of cycle 
      wait(Clk.posedge_event());
    } 
  } // end of while
} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_sync2::SCmdAcceptSetThread()
//
//  Set 'SCmdAccept' signal 
//  
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_sync2<TdataCl>::SCmdAcceptSetThread()
{
  SCmdAccept.write(true);
#ifdef DEBUG_G1
    cout << "TL0 Slave released request "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << endl;
#endif

} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_sync2::SCmdAcceptResetThread()
//
//  Reset 'SCmdAccept' signal 
//  
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_sync2<TdataCl>::SCmdAcceptResetThread()
{
  SCmdAccept.write(false);

} // end of method


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_sync2::ExecAndSendResponse()
//  
//  Reads requests in fifo
//  Acts as a memory for READ and WRITE operations (after 'ResponseDelay' time)
//  Sends OCP responses for WRITE operations
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_sync2<TdataCl>::ExecAndSendResponse()
{
  while(true) {

    if(m_request_fifo.num_available()==0) {
        wait();   // wait next clock cycle
    } else {

      // Read request from fifo
      request_struct<Td,Ta> req = m_request_fifo.read();

      // Test Memory Address Overflow 
      if((int)req.maddr >= MEMORY_SIZE) {
        cout << "Error: TL0 OCP Slave 'ExecAndSendResponse' method" << endl
          << "       Request Address exceeds Memory size" << endl
          << "       Request Address=" << (int)req.maddr
          << "       MEMORY_SIZE=" << MEMORY_SIZE
          << endl;
        assert(0);
      }

      // Read Operation
      if(req.mcmd == OCP_MCMD_RD) {

        // Response Delay data signal
        // This must work with or without the next line
        // The delay must be less than 1 clk cycle
         wait(m_ResponseDelay);

#ifdef DEBUG_G1
        cout << "TL0 Slave sent response "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds()
          << " addr " << (int)req.maddr
          << " data " << m_memory[(int)req.maddr] <<endl;
#endif

        // SResp & SData assertion 
        SResp.write(sc_bv<2>(OCP_SRESP_DVA)); 
        SData.write(m_memory[(int)req.maddr]);  

        // Wait end of current cycle and test 'MRespAccept'
        do {
          wait(Clk.posedge_event());
        } while ((MRespAccept.read() != true) );

        // SResp & SData de-assertion 
        SResp.write(sc_bv<2>(OCP_SRESP_NULL));
        SData.write(Td(0));

      } else if(req.mcmd == OCP_MCMD_WR) {
        // Write Operation
        m_memory[(int)req.maddr] = req.mdata;
#ifdef DEBUG_G1
        cout << "TL0 Slave Write Operation "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds()
          << " addr " << (int)req.maddr
          << " data " << req.mdata <<endl;
#endif
        wait(); // wait next clock cycle
      }

    } // end if 

  } // end of while

} // end of method


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------

template class OCP_TL0_Slave_sync2<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
