/*****************************************************************************

  The following code is derived, directly or indirectly, from the SystemC
  source code Copyright (c) 1996-2003 by all Contributors.
  All Rights reserved.

  The contents of this file are subject to the restrictions and limitations
  set forth in the SystemC Open Source License Version 2.4 (the "License");
  You may not use this file except in compliance with such restrictions and
  limitations. You may obtain instructions on how to receive a copy of the
  License at http://www.systemc.org/. Software distributed by Contributors
  under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
  ANY KIND, either express or implied. See the License for the specific
  language governing rights and limitations under the License.

 *****************************************************************************/

/*****************************************************************************

  MODIFICATION LOG - modifiers, enter your name, affiliation, date and
  changes you are making here.

      Name, Affiliation, Date:
  Description of Modification:

 *****************************************************************************/

// ============================================================================
//      Project : Generic SystemC TL Communication Methodology
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//         Date : 11/15/2002
//
//  Description : Transaction Level - Layer-3 example Master
//    This Master reads/writes ATM cells
//    Features: 
//      - Blocking methods are used
//      - Master is non-pipelined
//                
// Parameter    :
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//
// ============================================================================

#include "tl3_master.h"

// ----------------------------------------------------------------------------
// Process : TL3_Master::TL3_Master
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> TL3_Master<TdataCl>::TL3_Master
      (
        sc_module_name name_
      , int ID
      , bool WriteResponse
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_WriteResponse   (WriteResponse)
        , m_TaNum           (0)
        , m_TaNumS          (0)
{
  // InitParameters();

  m_Data = new Td[TL3_MAX_ATM_CELLS];

  SC_THREAD(Master);
}

// ----------------------------------------------------------------------------
// Process : TL3_Master::~TL3_Master
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> TL3_Master<TdataCl>::~TL3_Master()
{
  delete [] m_Data;
}


// ----------------------------------------------------------------------------
//  Method : TL3_Master::Master()
//
//  Top level method of the Master (non-pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Master<TdataCl>::Master()
{
  int i;
  int NumCells;
  bool Write;

  // we only need simple mechanism
  int jran = 0xDEAD + m_ID;

  // initialize Data array
  for (i = 0; i < TL3_MAX_ATM_CELLS; i++)
  {
    m_Data[i].GFC  = 10*i+0;
    m_Data[i].VPI  = 10*i+1;
    m_Data[i].VCI  = 10*i+2;
    m_Data[i].PTI  = 10*i+3;
    m_Data[i].CLP  = 10*i+4;
    m_Data[i].HEC  = 10*i+5;
    m_Data[i].Info = 10*i+6;
  }

  // send blocking requests
  while (true)
  {
    // Send different burst sizes for different Masters
    switch (m_ID)
    {
      case 1: NumCells = 2;
              break;
      case 2: NumCells = 4;
              break;
      case 3: NumCells = 8;
              break;
      default: NumCells = 2;
              break;
    }
    SetNumCells(NumCells);

    m_DataCl->MputNumCells(NumCells);
    m_DataCl->MputData(m_Data[0]); // explicit data copying

    // start with a few write requests:
    if (m_TaNum < TL3_MAX_ATM_CELLS / (2 * NumCells))
      Write = true;
    else
    {
      jran = (jran * 7141 + 54773) % 259200;
      Write = (((float)jran / 259200.0) < 0.5) ? 0 : 1;
    }

    if (Write)
    {
#ifdef DEBUG_G1
  PrintTime("Start write request in Master ", m_ID);
#endif
#ifdef DEBUG_M
  PrintTrans(m_TaNum, NumCells, " Blocking Write");
#endif

      // Send request to Bus/Slave
      if ((MasterP->MputWriteRequestBlocking()))
      {
        m_TaNum++;

        if (m_WriteResponse)
        {
          // Get response from Bus/Slave
          if ((MasterP->MgetResponseBlocking(true))) // release immediately
          {
            // get Slave return data from the Channel
            m_Sresp = m_DataCl->MgetStatus();

#ifdef DEBUG_M
  PrintSresp(m_Sresp, m_TaNum);
#endif
#ifdef DEBUG_G1
  PrintTime("End write response in Master ", m_ID);
#endif
          }
          else PrintError();
        }
      }
      else PrintError();
    } // end write branch
    else // read request
    {
#ifdef DEBUG_G1
  PrintTime("Start read request in Master ", m_ID);
#endif
#ifdef DEBUG_M
  PrintTrans(m_TaNum, NumCells, " Blocking Read");
#endif

      if ((MasterP->MputReadRequestBlocking()))
      {
        m_TaNum++;

        if ((MasterP->MgetResponseBlocking(true))) // release immediately
        {
          // get Slave return data from the Channel
          m_Sresp = m_DataCl->MgetStatus();

#ifdef DEBUG_M
  PrintSresp(m_Sresp, m_TaNum);
#endif
#ifdef DEBUG_G1
  PrintTime("Accept read response in Master ", m_ID);
#endif
        }
        else PrintError();
      }
      else PrintError();
    } // end read branch
  } // end while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : TL3_Master::SetNumCells()
//
//  Set the burst length
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool TL3_Master<TdataCl>::SetNumCells(
         int NumCells)

{
  if (NumCells > TL3_MAX_ATM_CELLS)
  {
    cerr << "ERROR: Master " << m_ID << " request exceeds max burst length. <"
         << name() << ">" << endl;
    sc_stop();
    return(false);
  }
  else
  {
    return(true);
  }
}


// ----------------------------------------------------------------------------
//  Method : TL3_Master::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool TL3_Master<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : TL3_Master::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Master<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//  Method : TL3_Master::PrintTrans()
//
//  Print transaction
//
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Master<TdataCl>::PrintTrans(
         int TaNum, int NumCells, const char *Method)

{
  cout << "Master " << m_ID << " : " << "Req = " << TaNum
	   << "  NumCells = " << NumCells << Method << endl;
}


// ----------------------------------------------------------------------------
//  Method : TL3_Master::PrintError()
//
//  
//
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Master<TdataCl>::PrintError(
         )

{
  cout << "Error: Master " << m_ID << " : " << "Req = " << m_TaNum << " failed"
       << endl;
}

// ----------------------------------------------------------------------------
//  Method : TL3_Master::PrintSresp()
//
//  
//
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Master<TdataCl>::PrintSresp(
         int Sresp, int TaNum)

{
  cout << "Master " << m_ID << " : " << "Req = " << TaNum << " Response = "
       << Sresp << endl;
}

// ----------------------------------------------------
//  Method : PrintTime()
//
//  Print time stamp
// ----------------------------------------------------
template<class TdataCl> void TL3_Master<TdataCl>::PrintTime(
         const char *Pstring, int ID)
{
  cout << Pstring << ID << " : " << endl;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------

#include "tl3_data_cl.h"
template class TL3_Master<TL3_TEMPL_DATA_CL >; // defined in tl3_globals.h
