// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Yann Bajot, Prosilog, bajot@prosilog.com
//         Date : 01/15/2004
//
//  Description : Transaction Level - Layer-1 example Master
//    Features:
//    - Asynchronous master using OCP2.0 data class
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//    - Performs 5 WRITE operations followed by 5 READ operations
//       and checks read data, assuming that the target
//       acts as a RAM
//                
// Parameter    :
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - RequestDelay: Non-negative number specifying the delay in percent
//                 of a clock cycle, the request signals are transmitted.
//     - ResponseDelay: Non-negative number specifying the delay in percent
//                 of a clock cycle, the response is acknowledged .
// ============================================================================

#include "ocp_tl1_master_async.h"
#include "ocp_tl1_data_cl.h"

// Used for debug purposes
#define DEBUG_TL1_MASTER_ASYNC

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async::OCP_TL1_Master_Async
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async<TdataCl>::OCP_TL1_Master_Async
      (
        sc_module_name name_
      , int ID
      , int Priority
      , double RequestDelay
      , double ResponseDelay
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_RequestDelay        (RequestDelay)
        , m_ResponseDelay        (ResponseDelay)
{
  // InitParameters();

  SC_THREAD(MasterRequest);
  sensitive_pos(clk); 
  SC_THREAD(MasterResponse);
  sensitive << MasterP; 
}

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async::~OCP_TL1_Master_Async
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async<TdataCl>::~OCP_TL1_Master_Async()
{
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::MasterRequest()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::MasterRequest()
{
  int i = 0;
  int j;
  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0}; // # of cycles between requests;
  //int Pause[] = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1}; // # of cycles between requests;
  //int Pause[] = {2, 2, 2, 2, 2, 2, 2, 2, 2, 2}; // # of cycles between requests;

  Ta Addr[] =     {0,4,8,12,16,0,4,8,12,16};    // MAddr field for each request
  Td SentData[] =     {5,4,3,2,1,0,0,0,0,0};    // MData field for each request
  int cnt = 0;

  // Reset the TL1 request structure
  TL1_request.reset();

  while(true)
  {
    wait(); // wait for rising edge of clock

    if (cnt == 0)
    {
      // check if channel is free
      if (!MasterP->getSBusy())
      {
        // get new Pause value
        j=i;
        i++;
        cnt = Pause[i%10];

        // Delay data signal
        // This must work with or without the next line
        if (m_RequestDelay > 0)
          wait(TL_CLK_PERIOD * (m_RequestDelay/100.0), TL_TIME_UNIT);

        // Put data into the channel
        TL1_request.MAddr = Addr[j%10];
        TL1_request.MData = SentData[j%10];
        
        if(j%10<5) {
          TL1_request.MCmd = OCP_MCMD_WR; // WRITE operations 
        }
        else {
          TL1_request.MCmd = OCP_MCMD_RD; // READ operations 
        }
        
        // Sent the TL1 request to the channel
        bool ret = MasterP->startOCPRequest(TL1_request);

        // For Debug purposes only
        assert(ret);


#ifdef DEBUG_TL1_MASTER_ASYNC
        cout << "TL1 Master sent request "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds()
          << " addr " << TL1_request.MAddr;
        if(j%10<5) 
          cout << " written data " << TL1_request.MData  << endl;
        else 
          cout << endl;
#endif

      }
    }
    else
    {
      cnt--;
    }
  } // end of while loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::MasterResponse()
//
//  Top level response method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::MasterResponse()
{
#ifdef DEBUG_TL1_MASTER_ASYNC
  Td CorrectData[] =  {5,4,3,2,1};
#endif

  int i = 0;

  while(true)
  {
    wait(); // for default event

    if (MasterP->getOCPResponse(TL1_response,false))
    {
      // There is a valid response on the channel

#ifdef DEBUG_TL1_MASTER_ASYNC
      cout << "TL1 Master got response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " received data " << TL1_response.SData ;
      if(TL1_response.SData != CorrectData[i%5]) 
        cout << " is BAD" << endl;
      else
        cout << " is good" << endl;
#endif
        
      // Models the response acknowledge delay
      if (m_ResponseDelay > 0)
        wait(TL_CLK_PERIOD * (m_ResponseDelay/100.0), TL_TIME_UNIT);

      // Release the response 
      MasterP->putMRespAccept();

#ifdef DEBUG_TL1_MASTER_ASYNC
      cout << "TL1 Master released response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << TL1_response.SData << endl;
#endif

      i++;
    } else {
        // No response available 
    }
  } // end of while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Master_Async<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Master_Async<OCP_TL1_DataCl<int,int>  >;
